import React, { useEffect, useRef, useState } from 'react';
import { View, Text, StyleSheet, Animated, Pressable, Easing } from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const CompostProgressBlock = () => {
  const [progress, setProgress] = useState(0);
  const animatedValue = useRef(new Animated.Value(0)).current;
  const fillAnim = useRef(new Animated.Value(0)).current;
  const timerRef = useRef(null);
  const [leaves, setLeaves] = useState([]);

  useEffect(() => {
    const loadProgress = async () => {
      const stored = await AsyncStorage.getItem('@compost_progress');
      const val = stored ? Number(stored) : 0;
      setProgress(val);
      animatedValue.setValue(val);
    };
    loadProgress();
  }, []);

  useEffect(() => {
    const interval = setInterval(() => {
      setProgress(prev => {
        const next = Math.min(prev + 1, 100);
        AsyncStorage.setItem('@compost_progress', String(next));
        Animated.timing(animatedValue, {
          toValue: next,
          duration: 500,
          useNativeDriver: false,
        }).start();
        return next;
      });
    }, 30000);
    return () => clearInterval(interval);
  }, []);

  useEffect(() => {
    const interval = setInterval(() => {
      spawnLeaf();
    }, Math.random() * 2000 + 2000);

    return () => clearInterval(interval);
  }, []);

  const spawnLeaf = () => {
    const id = Date.now();
    const translateY = new Animated.Value(0);
    const opacity = new Animated.Value(1);
    const rotate = new Animated.Value(0);
    const left = Math.random() * 80 + 10;

    const leaf = { id, translateY, opacity, rotate, left };
    setLeaves(current => [...current, leaf]);

    Animated.parallel([
      Animated.timing(translateY, {
        toValue: -60,
        duration: 4000,
        useNativeDriver: true,
        easing: Easing.out(Easing.quad),
      }),
      Animated.timing(opacity, {
        toValue: 0,
        duration: 4000,
        useNativeDriver: true,
      }),
      Animated.timing(rotate, {
        toValue: 1,
        duration: 4000,
        useNativeDriver: true,
        easing: Easing.linear,
      }),
    ]).start(() => {
      setLeaves(current => current.filter(l => l.id !== id));
    });
  };

  const handlePressIn = () => {
    fillAnim.setValue(0);
    Animated.timing(fillAnim, {
      toValue: 1,
      duration: 2000,
      useNativeDriver: false,
      easing: Easing.linear,
    }).start();

    timerRef.current = setTimeout(async () => {
      setProgress(0);
      animatedValue.setValue(0);
      await AsyncStorage.setItem('@compost_progress', '0');
    }, 2000);
  };

  const handlePressOut = () => {
    clearTimeout(timerRef.current);
    fillAnim.stopAnimation();
    fillAnim.setValue(0);
  };

  const animatedStyle = {
    width: animatedValue.interpolate({
      inputRange: [0, 100],
      outputRange: ['0%', '100%'],
    }),
  };

  const fillStyle = {
    width: fillAnim.interpolate({
      inputRange: [0, 1],
      outputRange: ['0%', '100%'],
    }),
  };

  return (
    <Pressable onPressIn={handlePressIn} onPressOut={handlePressOut} style={styles.container}>
      <View style={styles.progressContainer}>
        <Animated.View style={[styles.fill, animatedStyle]} />
        <Animated.View style={[styles.fillReset, fillStyle]} />
        {leaves.map(leaf => {
          const rotateInterpolate = leaf.rotate.interpolate({
            inputRange: [0, 1],
            outputRange: ['0deg', '90deg'],
          });
          return (
            <Animated.View
              key={leaf.id}
              style={[
                styles.leaf,
                {
                  transform: [
                    { translateY: leaf.translateY },
                    { rotate: rotateInterpolate },
                  ],
                  opacity: leaf.opacity,
                  left: `${leaf.left}%`,
                },
              ]}
            />
          );
        })}
        {progress === 100 ? (
          <View style={styles.messageContainer}>
            <Text style={styles.labelBold}>100% OPEN THE HATCH</Text>
            <Text style={styles.labelSmall}>hold 3s. to reset</Text>
          </View>
        ) : (
          <Text style={styles.label}>{progress}% completed</Text>
        )}
      </View>
    </Pressable>
  );
};

const styles = StyleSheet.create({
  container: {
    borderRadius: 16,
    overflow: 'hidden',
    width: '90%',
    height: 80,
    backgroundColor: '#D9D9D9',
    justifyContent: 'center',
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 3,
  },
  progressContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  fill: {
    ...StyleSheet.absoluteFillObject,
    backgroundColor: '#BBBD9A',
  },
  fillReset: {
    ...StyleSheet.absoluteFillObject,
    backgroundColor: '#c7c3d4',
  },
  label: {
    textAlign: 'center',
    fontWeight: 'bold',
    fontSize: 18,
    color: 'white',
  },
  labelBold: {
    textAlign: 'center',
    fontWeight: 'bold',
    fontSize: 18,
    color: 'white',
  },
  labelSmall: {
    textAlign: 'center',
    fontSize: 12,
    color: 'white',
    opacity: 0.8,
  },
  messageContainer: {
    alignItems: 'center',
    justifyContent: 'center',
  },
  leaf: {
    position: 'absolute',
    bottom: 0,
    width: 12,
    height: 18,
    backgroundColor: '#4CAF50',
    borderRadius: 10,
    opacity: 0.7,
  },
});

export default CompostProgressBlock;
