import React, { useEffect, useRef, useState } from 'react';
import { View, Text, TouchableOpacity, Animated, Dimensions, ActivityIndicator, Easing, Image } from 'react-native';
import { logout } from '../auth';
import styles from '../styles/styles';
import CompostProgressBar from './Bar';
import AsyncStorage from '@react-native-async-storage/async-storage';

export default function HomeScreen({ navigation }) {
  const scrollX = useRef(new Animated.Value(0)).current;
  const screenWidth = Dimensions.get('window').width;
  const [username, setUsername] = useState('anonymous');
  const [status, setStatus] = useState(null);
  const [loadingStatus, setLoadingStatus] = useState(true);
  const [isFanActive, setIsFanActive] = useState(false);

  const fanRotation = useRef(new Animated.Value(0)).current;

  const fanSpinStyle = {
    transform: [
      {
        rotate: fanRotation.interpolate({
          inputRange: [0, 1],
          outputRange: ['0deg', '360deg'],
        }),
      },
    ],
  };

  useEffect(() => {
    const anim = Animated.loop(
      Animated.timing(scrollX, {
        toValue: -screenWidth,
        duration: 10000,
        useNativeDriver: true
      })
    );
    anim.start();
    return () => anim.stop();
  }, []);

  useEffect(() => {
    const fetchUsername = async () => {
      const storedUsername = await AsyncStorage.getItem('@username');
      if (storedUsername) setUsername(storedUsername);
    };
    fetchUsername();
  }, []);

  useEffect(() => {
    const fetchStatus = async () => {
      try {
        const res = await fetch('http://51.210.149.28:5000/api/home-status');
        const json = await res.json();
        setStatus(json);
      } catch (err) {
        console.error('Status fetch error:', err);
      } finally {
        setLoadingStatus(false);
      }
    };
    fetchStatus();
  }, []);
  
  useEffect(() => {
    const fetchFanState = async () => {
      try {
        const res = await fetch('http://51.210.149.28:5000/api/fan');
        const json = await res.json();
        setIsFanActive(json.state === 'on');
      } catch (err) {
        console.error('Erreur fetch fan state', err);
      }
    };

    fetchFanState(); 
    const interval = setInterval(fetchFanState, 3000);
    return () => clearInterval(interval);
  }, []);

  useEffect(() => {
    if (isFanActive) {
      spinFan();
    } else {
      stopFanSpin();
    }
  }, [isFanActive]);

  const spinFan = () => {
    fanRotation.setValue(0);
    Animated.loop(
      Animated.timing(fanRotation, {
        toValue: 1,
        duration: 2000,
        easing: Easing.linear,
        useNativeDriver: true,
      })
    ).start();
  };

  const stopFanSpin = () => {
    fanRotation.stopAnimation();
  };

  const toggleFan = async () => {
    const newState = !isFanActive;

    try {
      await fetch(`http://51.210.149.28:5000/api/fan`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ state: newState ? 'on' : 'off' })
      });

      setIsFanActive(newState);
    } catch (err) {
      console.error('Fan toggle error:', err);
    }
  };

  const handleLogout = async () => {
    await AsyncStorage.removeItem('@username');
    await logout();
    navigation.replace('Login');
  };

  const getStyleForMessage = (msg) => {
    if (!msg) return styles.statusText;
    if (msg.toLowerCase().includes("ok")) return styles.statusTextOK;
    if (msg.toLowerCase().includes("ajoutez") || msg.toLowerCase().includes("mélanger") || msg.toLowerCase().includes("ventilez")) return styles.statusTextWarning;
    if (msg.toLowerCase().includes("important") || msg.toLowerCase().includes("élevé")) return styles.statusTextDanger;
    return styles.statusText;
  };

  return (
    <View style={styles.container}>

      <Image source={require('../assets/loopbin.png')} style={styles.logoSmall} />

      <TouchableOpacity style={styles.SUSButton} onPress={() => navigation.navigate('Survey', { username })}>
        <Text style={styles.logoutText}>SUS Test</Text>
      </TouchableOpacity>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.mainButton} onPress={() => navigation.navigate('Sensors')}>
          <Text style={styles.buttonText}>Layer 1</Text>
          {loadingStatus ? (
            <ActivityIndicator style={{ marginTop: 10 }} />
          ) : (
            <Text style={getStyleForMessage(status?.layer1_status)}>{status?.layer1_status || 'No data'}</Text>
          )}
        </TouchableOpacity>

        <TouchableOpacity style={styles.BButton} onPress={() => navigation.navigate('Sensors2')}>
          <Text style={styles.buttonText}>Layer 2</Text>
          {loadingStatus ? (
            <ActivityIndicator style={{ marginTop: 10 }} />
          ) : (
            <Text style={getStyleForMessage(status?.layer2_status)}>{status?.layer2_status || 'No data'}</Text>
          )}
        </TouchableOpacity>

        <CompostProgressBar />

        <TouchableOpacity style={styles.fanButton} onPress={toggleFan}>
          <Animated.Image
            source={require('../assets/fan.png')}
            style={[styles.fanIcon, fanSpinStyle]}
          />
          <Text style={styles.buttonText}>{isFanActive ? 'Turn off' : 'Turn on'}</Text>
        </TouchableOpacity>
      </View>

      <TouchableOpacity style={styles.logoutButton} onPress={handleLogout}>
        <Text style={styles.logoutText}>Disconnect</Text>
      </TouchableOpacity>

      <View style={styles.banner}>
        <Animated.Text style={[styles.bannerText, { transform: [{ translateX: scrollX }], width: screenWidth * 2 }]}>
          Please test the app, then complete SUS Test :)
        </Animated.Text>
      </View>
    </View>
  );
}
