import React, { useState } from 'react';
import { View, Text, TouchableOpacity, ScrollView, StyleSheet, Alert } from 'react-native';

const STRINGS = {
  en: {
    title: "SUS Survey",
    submit: "Submit",
    incomplete: "Please answer all questions.",
    thankYou: "Thank you!",
    submitted: "Feedback submitted.\nSUS Score: ",
    error: "Could not send survey: ",
    questions: [
      "I think that I would like to use this app frequently.",
      "I found the app unnecessarily complex.",
      "I thought the app was easy to use.",
      "I think that I would need help to use this app.",
      "I found the various functions well integrated.",
      "I thought there was too much inconsistency.",
      "I would imagine most people would learn to use this app very quickly.",
      "I found the app very cumbersome to use.",
      "I felt very confident using the app.",
      "I needed to learn a lot of things before I could get going."
    ]
  },
  fr: {
    title: "Questionnaire SUS",
    submit: "Envoyer",
    incomplete: "Veuillez répondre à toutes les questions.",
    thankYou: "Merci !",
    submitted: "Avis enregistré.\nScore SUS : ",
    error: "Impossible d'envoyer le questionnaire : ",
    questions: [
      "Je pense que j’aimerais utiliser cette app fréquemment.",
      "J’ai trouvé l’app inutilement complexe.",
      "J’ai trouvé l’app facile à utiliser.",
      "Je pense que j’aurais besoin d’aide pour utiliser cette app.",
      "J’ai trouvé les fonctions bien intégrées.",
      "J’ai trouvé trop d’incohérences dans l’app.",
      "Je pense que la plupart des gens apprendraient à utiliser cette app très vite.",
      "J’ai trouvé l’app lourde à utiliser.",
      "Je me suis senti(e) en confiance avec l’app.",
      "J’ai dû apprendre beaucoup de choses avant de commencer."
    ]
  }
};

export default function SurveyScreen({ route, navigation }) {
  const [answers, setAnswers] = useState(Array(10).fill(null));
  const [lang, setLang] = useState('en');
  const t = STRINGS[lang];
  const username = route.params?.username || 'anonymous';

  const handleSelect = (questionIndex, value) => {
    const newAnswers = [...answers];
    newAnswers[questionIndex] = value;
    setAnswers(newAnswers);
  };

  const handleSubmit = async () => {
    if (answers.some(a => typeof a !== 'number')) {
      Alert.alert(t.incomplete);
      return;
    }

    try {
      const response = await fetch('http://51.210.149.28:5000/api/survey', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ username, responses: answers })
      });

      const json = await response.json();
      if (!response.ok) throw new Error(json.error || 'Unknown error');

      Alert.alert(t.thankYou, t.submitted + json.score);
      navigation.goBack();
    } catch (e) {
      console.error('Survey error:', e);
      Alert.alert('Error', t.error + e.message);
    }
  };

  return (
    <ScrollView contentContainerStyle={styles.container}>
      <TouchableOpacity onPress={() => setLang(lang === 'en' ? 'fr' : 'en')}>
        <Text style={styles.langSwitch}>
          {lang === 'en' ? '🇫🇷 Version française' : '🇬🇧 English version'}
        </Text>
      </TouchableOpacity>

      <Text style={styles.title}>{t.title}</Text>

      {t.questions.map((q, index) => (
        <View key={index} style={styles.questionBlock}>
          <Text style={styles.question}>{index + 1}. {q}</Text>
          <View style={styles.scaleContainer}>
            {[1, 2, 3, 4, 5].map((val) => (
              <TouchableOpacity
                key={val}
                style={[
                  styles.numberCircle,
                  answers[index] === val && styles.selectedCircle
                ]}
                onPress={() => handleSelect(index, val)}
              >
                <Text style={[
                  styles.numberText,
                  answers[index] === val && styles.selectedText
                ]}>{val}</Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>
      ))}

      <TouchableOpacity style={styles.submitButton} onPress={handleSubmit}>
        <Text style={styles.submitText}>{t.submit}</Text>
      </TouchableOpacity>
    </ScrollView>
  );
}

const styles = StyleSheet.create({
  container: {
    padding: 20,
    paddingBottom: 50
  },
  langSwitch: {
    textAlign: 'right',
    marginBottom: 10,
    color: '#888',
    fontStyle: 'italic',
    marginTop:30
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    marginBottom: 20,
    textAlign: 'center',
    color: '#BBBD9A'
  },
  questionBlock: {
    marginBottom: 25
  },
  question: {
    fontSize: 16,
    marginBottom: 10
  },
  scaleContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around'
  },
  numberCircle: {
    width: 40,
    height: 40,
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#999',
    alignItems: 'center',
    justifyContent: 'center'
  },
  selectedCircle: {
    backgroundColor: '#BBBD9A',
    borderColor: '#BBBD9A'
  },
  numberText: {
    fontSize: 16,
    color: '#333'
  },
  selectedText: {
    color: '#fff',
    fontWeight: 'bold'
  },
  submitButton: {
    marginTop: 30,
    backgroundColor: '#BBBD9A',
    padding: 15,
    borderRadius: 8,
    alignItems: 'center'
  },
  submitText: {
    color: 'white',
    fontWeight: 'bold',
    fontSize: 16
  }
});
