from flask import Flask, request, jsonify, render_template_string
from datetime import datetime, timedelta
import pytz
import json
import os
from werkzeug.security import generate_password_hash, check_password_hash


app = Flask(__name__)

last_data = {
    # Layer 1
    "temperature": 0.0,
    "humidity": 0.0,
    "co2": 0,
    # Layer 2
    "temperature2": 0.0,
    "humidity2": 0.0,
    "co22": 0,
    "timestamp": "--"
}

etat = "pause"
last_measure_time = None  
fan_state = False

HTML_TEMPLATE = """<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <title>LoopBin Sensors</title>
  <link rel="stylesheet" href="{{ url_for('static', filename='css/style.css') }}">
</head>
<body>

  <h1>LoopBin Sensors</h1>

  <div class="card">
    <div class="sensor-value">
      <span class="label">Temperature :</span><br>
      <span id="temperature">{{ temperature }}</span> °C<br>

      <span class="label">Moisture :</span><br>
      <span id="humidity">{{ humidity }}</span> %<br>

      <span class="label">CO<sub>2</sub> :</span><br>
      <span id="co2">{{ co2 }}</span> kPPM
    </div>

    <p><em>Last update <span id="timestamp">{{ timestamp }}</span></em></p>
  </div>

  <!-- Sliders and buttons omitted for brevity -->

  <script>
    function triggerMeasure() {
      fetch('/api/trigger')
        .then(res => res.json())
        .then(data => {
          console.log(data);
          setTimeout(refreshData, 2000);
        })
        .catch(err => console.error('Erreur déclenchement:', err));
    }

    function refreshData() {
      fetch('/api/data/layer1')  // By default fetch layer1 data for the web page
        .then(response => response.json())
        .then(data => {
          document.getElementById('temperature').textContent = data.temperature.toFixed(2);
          document.getElementById('humidity').textContent = data.humidity.toFixed(2);
          document.getElementById('co2').textContent = data.co2.toFixed(0);
          document.getElementById('timestamp').textContent = data.timestamp;

          // Update sliders logic here (omitted for brevity)
        })
        .catch(error => console.error('Erreur récupération données:', error));
    }

    window.onload = refreshData;
    setInterval(refreshData, 5000);
  </script>

</body>
</html>"""

# Routage
@app.route('/')
def home():
    return render_template_string(HTML_TEMPLATE, **last_data)

@app.route('/api/data/layer1', methods=['GET'])
def get_layer1_data():
    data = {
        "temperature": last_data.get("temperature", 0),
        "humidity": last_data.get("humidity", 0),
        "co2": last_data.get("co2", 0),
        "timestamp": last_data.get("timestamp", "--")
    }
    return jsonify(data)

@app.route('/api/data/layer2', methods=['GET'])
def get_layer2_data():
    data = {
        "temperature": last_data.get("temperature2", 0),
        "humidity": last_data.get("humidity2", 0),
        "co2": last_data.get("co22", 0),
        "timestamp": last_data.get("timestamp", "--")
    }
    return jsonify(data)

@app.route('/api/etat', methods=['GET'])
def get_etat():
    global etat, last_measure_time
    now = datetime.now(pytz.timezone('Europe/Lisbon'))

    if (last_measure_time is None) or (now - last_measure_time > timedelta(minutes=1)):
        etat = "demande"
    else:
        etat = "pause"

    return jsonify({"etat": etat})

@app.route('/api/post-data', methods=['POST'])
def recevoir_donnees():
    global last_data, etat, last_measure_time
    data = request.get_json()
    if not data:
        return jsonify({"error": "Aucune donnée reçue"}), 400
    try:
        
        if "temperature" in data:
            last_data["temperature"] = float(data.get("temperature", 0))
        if "humidity" in data:
            last_data["humidity"] = float(data.get("humidity", 0))
        if "co2" in data or "Co2" in data or "CO2" in data:
            last_data["co2"] = int(data.get("co2") or data.get("Co2") or data.get("CO2") or 0)


        if "temperature2" in data:
            last_data["temperature2"] = float(data.get("temperature2", 0))
        if "humidity2" in data:
            last_data["humidity2"] = float(data.get("humidity2", 0))
        if "co22" in data:
            last_data["co22"] = int(data.get("co22", 0))

        porto_tz = pytz.timezone('Europe/Lisbon')
        now = datetime.now(porto_tz)
        last_data["timestamp"] = now.strftime("%d/%m/%Y %H:%M:%S")

        last_measure_time = now
        etat = "pause"

        print(f"Nouvelle donnée reçue : {last_data}")
        return jsonify({"status": "ok"})
    except Exception as e:
        return jsonify({"error": str(e)}), 400

@app.route('/api/trigger', methods=['GET'])
def trigger():
    global etat
    etat = "demande"
    return jsonify({"status": "ok"})


USERS_FILE = '../users.json'

def load_users():
    if os.path.exists(USERS_FILE):
        with open(USERS_FILE, 'r') as f:
            return json.load(f)
    return {}

def save_users(users):
    with open(USERS_FILE, 'w') as f:
        json.dump(users, f)

@app.route('/api/signup', methods=['POST'])
def signup():
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')

    if not username or not password:
        return jsonify({'success': False, 'message': 'Champs manquants'}), 400

    users = load_users()

    if username in users:
        return jsonify({'success': False, 'message': 'Utilisateur existe déjà'}), 409

    users[username] = generate_password_hash(password)
    save_users(users)
    return jsonify({'success': True, 'message': 'Inscription réussie'})

@app.route('/api/login', methods=['POST'])
def login():
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')

    users = load_users()

    if username not in users:
        return jsonify({'success': False, 'message': 'Utilisateur inconnu'}), 404

    if check_password_hash(users[username], password):
        return jsonify({'success': True})
    else:
        return jsonify({'success': False, 'message': 'Mot de passe incorrect'}), 401


DATA_FILE = '../sus_results.json'

def calculate_sus_score(responses):
    if len(responses) != 10:
        raise ValueError("There must be exactly 10 responses")

    adjusted = []
    for i, score in enumerate(responses):
        if not 1 <= score <= 5:
            raise ValueError("Each response must be between 1 and 5")
        if i % 2 == 0:
            adjusted.append(score - 1)
        else:
            adjusted.append(5 - score)

    return sum(adjusted) * 2.5

def load_results():
    if os.path.exists(DATA_FILE):
        with open(DATA_FILE, 'r') as f:
            return json.load(f)
    return {}

def save_results(results):
    with open(DATA_FILE, 'w') as f:
        json.dump(results, f, indent=2)

@app.route('/api/survey', methods=['POST'])
def submit_survey():
    data = request.get_json()
    username = data.get('username')
    responses = data.get('responses')

    if not username or not isinstance(responses, list):
        return jsonify({'error': 'Invalid data'}), 400

    try:
        score = calculate_sus_score(responses)
    except ValueError as e:
        return jsonify({'error': str(e)}), 400

    results = load_results()
    results[username] = score
    save_results(results)

    return jsonify({'message': 'Survey saved', 'score': score}), 200

def generate_status(temp, humidity, co2):
    alerts = []

    # Température
    if temp > 60:
        alerts.append("High temperature – ventilate")
    elif temp < 50:
        alerts.append("Low temperature – mix")

    # Humidité
    if humidity < 55:
        alerts.append("Compost too dry – add green waste")
    elif humidity > 65:
        alerts.append("Compost too wet – add brown waste")

    # CO2
    if co2 > 150:
        alerts.append("CO2 level too high – ventilate")

    # Si aucune alerte
    if not alerts:
        return "Everything is perfect :)"

    return " | ".join(alerts)

@app.route('/api/home-status', methods=['GET'])
def home_status():
    layer1_msg = generate_status(
        last_data.get("temperature", 0),
        last_data.get("humidity", 0),
        last_data.get("co2", 0)
    )

    layer2_msg = generate_status(
        last_data.get("temperature2", 0),
        last_data.get("humidity2", 0),
        last_data.get("co22", 0)
    )

    return jsonify({
        "layer1_status": layer1_msg,
        "layer2_status": layer2_msg
    })

@app.route('/api/fan', methods=['GET'])
def get_fan_state():
    global fan_state
    return jsonify({"state": "on" if fan_state else "off"})


@app.route('/api/fan', methods=['POST'])
def set_fan_state():
    global fan_state
    data = request.get_json()
    if not data or 'state' not in data:
        return jsonify({"error": "Missing 'state' parameter"}), 400

    state = data['state'].lower()
    if state not in ['on', 'off']:
        return jsonify({"error": "Invalid state, must be 'on' or 'off'"}), 400

    fan_state = (state == 'on')
    print(f"[INFO] Ventilateur mis à {'ON' if fan_state else 'OFF'} par un utilisateur.")
    return jsonify({"state": state})

if __name__ == "__main__":
    app.run(host="0.0.0.0", port=5000, debug=True)
