import React, { useEffect, useState } from 'react';
import { View, Text, ScrollView, RefreshControl, StyleSheet, Image, TouchableOpacity, Dimensions, SafeAreaView } from 'react-native';
import { useNavigation } from '@react-navigation/native';

const API_URL = 'http://51.210.149.28:5000/api/data/layer2';

const SENSOR_ORDER = ['temperature', 'humidity', 'co2'];
const LABELS = {
  temperature: 'Temperature',
  humidity: 'Moisture',
  co2: "CO\u2082"
};

const UNITS = {
  temperature: '°C',
  humidity: '%',
  co2: 'kppm'
};

export default function SensorScreen() {
  const navigation = useNavigation();
  const [data, setData] = useState({});
  const [refreshing, setRefreshing] = useState(false);

  const fetchData = async () => {
    try {
      const response = await fetch(API_URL);
      const json = await response.json();
      setData(json);
    } catch (error) {
      console.error('Erreur de récupération des données:', error);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  const onRefresh = async () => {
    setRefreshing(true);
    await fetchData();
    setRefreshing(false);
  };

  const lastUpdated = data.timestamp ? data.timestamp : 'N/A';

  return (
    <SafeAreaView style={{ flex: 1 }}>
      <ScrollView
        contentContainerStyle={styles.container}
        refreshControl={<RefreshControl refreshing={refreshing} onRefresh={onRefresh} />}
        style={{ flex: 1 }}
      >
        <Text style={styles.title}>Layer 2</Text>

        {SENSOR_ORDER.map((key) => (
          <View key={key} style={styles.card}>
            <Text style={styles.cardTitle}>{LABELS[key]}</Text>
            <Text style={styles.cardValue}>
              {data[key] !== undefined ? `${data[key]} ${UNITS[key]}` : '...'}
            </Text>
          </View>
        ))}

        <Text style={styles.timestamp}>Last refresh : {lastUpdated}</Text>

        <TouchableOpacity onPress={() => navigation.navigate('Home')} style={styles.logoButton}>
          <Image source={require('../assets/homeg.png')} style={styles.logoImage} />
        </TouchableOpacity>
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    marginBottom: 150,
    textAlign: 'center',
    marginTop: 50,
  },
  container: {
    padding: 20,
    alignItems: 'center',
    paddingBottom: 60,
  },
  card: {
    backgroundColor: '#c7c3d4',
    padding: 20,
    borderRadius: 10,
    marginBottom: 15,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 5,
    elevation: 2,
    width: '100%',
  },
  cardTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 5,
  },
  cardValue: {
    fontSize: 22,
    fontWeight: 'bold',
    color: '#FFF',
  },
  timestamp: {
    marginTop: 30,
    fontStyle: 'italic',
    color: '#666',
  },
  logoButton: {
    marginTop: 40,
    alignItems: 'center',
  },
  logoImage: {
    width: 80,
    height: 80,
    resizeMode: 'contain',
  },
});
