import React, { useEffect, useState, useRef } from 'react';
import { View, Text, ScrollView, RefreshControl, StyleSheet, Image, TouchableOpacity, Animated, Dimensions } from 'react-native';
import { useNavigation } from '@react-navigation/native';

const API_URL = 'http://51.210.149.28:5000/api/data';

const SENSOR_ORDER = ['temperature', 'humidity', 'co2'];
const LABELS = {
  temperature: 'Temperature',
  humidity: 'Moisture',
  co2: "CO\u2082"
};

const UNITS = {
  temperature: '°C',
  humidity: '%',
  co2: 'ppm'
};

export default function SensorScreen() {
  const scrollX = useRef(new Animated.Value(0)).current;  
  const screenWidth = Dimensions.get('window').width;
  const navigation = useNavigation(); 
  const [data, setData] = useState({});
  const [refreshing, setRefreshing] = useState(false);

  const fetchData = async () => {
    try {
      const response = await fetch(API_URL);
      const json = await response.json();
      setData(json);
    } catch (error) {
      console.error('Erreur de récupération des données:', error);
    }
  };

  useEffect(() => {
    const anim = Animated.loop(
      Animated.timing(scrollX, { toValue: -screenWidth, duration: 10000, useNativeDriver: true })
    );
    anim.start();
    return () => anim.stop();
  }, []);

  useEffect(() => {
    fetchData();
  }, []);

  const onRefresh = async () => {
    setRefreshing(true);
    await fetchData();
    setRefreshing(false);
  };

  const lastUpdated = data.timestamp ? data.timestamp : 'N/A';

  return (
    <ScrollView
      contentContainerStyle={styles.container}
      refreshControl={<RefreshControl refreshing={refreshing} onRefresh={onRefresh} />}
    >
      <Text style={styles.title}>Layer 2</Text>

      {SENSOR_ORDER.map((key) => (
        <View key={key} style={styles.card}>
          <Text style={styles.cardTitle}>{LABELS[key]}</Text>
          <Text style={styles.cardValue}>
            {data[key] !== undefined ? `${data[key]} ${UNITS[key]}` : '...'}
          </Text>
        </View>
      ))}

      <Text style={styles.timestamp}>Last refresh : {lastUpdated}</Text>

      <TouchableOpacity onPress={() => navigation.navigate('Home')} style={styles.logoButton}>
        <Image source={require('../assets/home.png')} style={styles.logoImage} />
      </TouchableOpacity>
      <View style={styles.banner}>
        <Animated.Text style={[styles.bannerText, { transform: [{ translateX: scrollX }], width: screenWidth * 2 }]}>
        Sensors of layer 1 used here, add more sensors for deferents values !
        </Animated.Text>
      </View>
    </ScrollView>
  );
}

const styles = StyleSheet.create({
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    marginBottom: 150,
    textAlign: 'center',
    marginTop: 50
  },
  container: {
    padding: 20,
    alignItems: 'center'
  },
  card: {
    backgroundColor: '#c7c3d4',
    padding: 20,
    borderRadius: 10,
    marginBottom: 15,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 5,
    elevation: 2,
    width: '100%'
  },
  cardTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 5
  },
  cardValue: {
    fontSize: 22,
    fontWeight: 'bold',
    color: '#FFF'
  },
  timestamp: {
    marginTop: 30,
    fontStyle: 'italic',
    color: '#666'
  },
  logoButton: {
    marginTop: 40,
    alignItems: 'center'
  },
  logoImage: {
    width: 80,
    height: 80,
    resizeMode: 'contain'
  },
  banner: 
  { backgroundColor: '#BBBD9A', position: 'absolute', bottom: 0, width: '100%', height: 50, justifyContent: 'center', overflow: 'hidden', paddingLeft: 10 },

  bannerText: 
  { fontSize: 16, fontWeight: 'bold', whiteSpace: 'nowrap' }
});
