import React, { useEffect, useState } from 'react';
import { View, Text, ScrollView, RefreshControl, StyleSheet } from 'react-native';

const API_URL = 'http://51.210.149.28:5000/api/data'; 

// Ordre + labels lisibles
const SENSOR_ORDER = ['temperature', 'humidity', 'co2'];
const LABELS = {
  temperature: 'Temperature',
  humidity: 'Moisture',
  co2: "CO\u2082"
};

export default function SensorScreen() {
  const [data, setData] = useState({});
  const [refreshing, setRefreshing] = useState(false);

  const fetchData = async () => {
    try {
      const response = await fetch(API_URL);
      const json = await response.json();
      setData(json);
    } catch (error) {
      console.error('Erreur de récupération des données:', error);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  const onRefresh = async () => {
    setRefreshing(true);
    await fetchData();
    setRefreshing(false);
  };

  const lastUpdated = data.timestamp ? data.timestamp : 'N/A';
  
  return (
    <ScrollView
      contentContainerStyle={styles.container} refreshControl={<RefreshControl refreshing={refreshing} onRefresh={onRefresh} />}>
    <Text style={styles.title}>Layer 1</Text>
    
      {SENSOR_ORDER.map((key) => (
        <View key={key} style={styles.card}>
          <Text style={styles.cardTitle}>{LABELS[key]}</Text>
          <Text style={styles.cardValue}>{data[key] || '...'}</Text>
        </View>
      ))}

      <Text style={styles.timestamp}>Last refresh : {lastUpdated}</Text>
    </ScrollView>
  );
}

const styles = StyleSheet.create({
  title: { fontSize: 24, fontWeight: 'bold', marginBottom: 150, textAlign: 'center',marginTop: 50},
  container: { padding: 20, alignItems: 'center' },
  card: { backgroundColor: '#BBBD9A', padding: 20, borderRadius: 10, marginBottom: 15, shadowColor: '#000', shadowOpacity: 0.1, shadowRadius: 5, elevation: 2 },
  cardTitle: { fontSize: 18, fontWeight: 'bold', marginBottom: 5 },
  cardValue: { fontSize: 22, fontWeight: 'bold', color: '#FFF' },
  timestamp: { marginTop: 30, fontStyle: 'italic', color: '#666' }
});
